// 全局错误处理器 - 项目级别的401拦截机制
import { getRouter } from './router-manager'

// 防止重复跳转的标志
let isHandling401 = false

/**
 * 处理401错误 - 项目级别拦截
 */
export const handle401Error = (errorMessage = '未授权访问，请重新登录') => {
  // 检查是否已经在处理中，防止无限循环
  if (isHandling401) {
    console.log('已在处理401错误，避免重复操作')
    return
  }
  
  // 设置处理标志
  isHandling401 = true
  
  console.log('执行全局401错误处理')
  
  // 清除本地存储的用户信息
  localStorage.removeItem('wechat_user')
  
  // 获取路由实例
  const router = getRouter()
  
  // 跳转到登录页
  if (router) {
    // 使用replace而不是push，避免浏览器历史记录问题
    router.replace('/login')
  } else {
    // 备用方案：直接修改URL
    if (window.location.hash) {
      window.location.hash = '/login'
    } else {
      window.location.pathname = '/login'
    }
  }
  
  // 3秒后重置处理标志
  setTimeout(() => {
    isHandling401 = false
    console.log('重置401处理标志')
  }, 3000)
  
  // 抛出错误以便上层可以捕获
  throw new Error(errorMessage)
}

/**
 * 检查是否为401错误
 */
export const is401Error = (error) => {
  if (!error) return false
  
  // 检查axios错误
  if (error.response && error.response.status === 401) {
    return true
  }
  
  // 检查fetch错误
  if (error.status === 401) {
    return true
  }
  
  // 检查错误消息
  if (error.message && (
    error.message.includes('401') ||
    error.message.includes('未授权') ||
    error.message.includes('unauthorized')
  )) {
    return true
  }
  
  return false
}

/**
 * 全局错误拦截器
 */
export const setupGlobalErrorInterceptor = () => {
  // 拦截控制台错误
  const originalConsoleError = console.error
  console.error = (...args) => {
    // 检查是否包含401相关错误
    const errorString = args.join(' ')
    if (errorString.includes('401') || errorString.includes('未授权')) {
      console.log('控制台检测到401错误:', errorString)
    }
    originalConsoleError.apply(console, args)
  }
  
  // 拦截未捕获的Promise错误
  window.addEventListener('unhandledrejection', (event) => {
    const error = event.reason
    if (is401Error(error)) {
      console.log('未捕获的Promise错误中包含401:', error)
      event.preventDefault() // 阻止默认行为
      handle401Error()
    }
  })
  
  // 拦截未捕获的JavaScript错误
  window.addEventListener('error', (event) => {
    const error = event.error
    if (is401Error(error)) {
      console.log('未捕获的JavaScript错误中包含401:', error)
      event.preventDefault() // 阻止默认行为
      handle401Error()
    }
  })
}

/**
 * 增强的fetch拦截器
 */
export const setupFetchInterceptor = () => {
  const originalFetch = window.fetch
  
  if (originalFetch) {
    window.fetch = async (...args) => {
      try {
        const response = await originalFetch(...args)
        
        // 检查401状态码
        if (response.status === 401) {
          console.log('Fetch请求返回401，执行全局处理')
          handle401Error()
        }
        
        return response
      } catch (error) {
        if (is401Error(error)) {
          handle401Error()
        }
        throw error
      }
    }
  }
}

/**
 * 初始化全局错误处理
 */
export const initGlobalErrorHandler = () => {
  setupGlobalErrorInterceptor()
  setupFetchInterceptor()
  console.log('全局错误处理机制已初始化')
}

export default {
  handle401Error,
  is401Error,
  setupGlobalErrorInterceptor,
  setupFetchInterceptor,
  initGlobalErrorHandler
}