import { EventSourcePolyfill } from 'event-source-polyfill';
import { ref, Ref } from 'vue';

// SSE数据类型定义
export interface SSEData {
  message: any;
  status: number | string;
  type: number | string;
}

// SSE服务配置
export interface SSEServiceConfig {
  apiBaseUrl: string;
  appCode: string;
  token: string;
  params: {
    stage?: string;
    appId?: string;
  };
}

// SSE事件处理器
export interface SSEHandlers {
  onMessage?: (data: SSEData) => void;
  onError?: (error: any) => void;
  onOpen?: (event: any) => void;
  onReconnect?: (newDialogSessionId: string) => void;
}

// SSE服务类 - 专注于SSE连接管理
export class SSEService {
  private eventSource: EventSourcePolyfill | null = null;
  private config: SSEServiceConfig;
  private handlers: SSEHandlers;
  private isReconnecting: Ref<boolean> = ref(false);
  private timeArr: NodeJS.Timeout[] = [];

  constructor(config: SSEServiceConfig, handlers: SSEHandlers = {}) {
    this.config = config;
    this.handlers = handlers;
  }

  // 初始化SSE连接
  public initSSE(dialogSessionId: string): void {
    try {
      const url = `${this.config.apiBaseUrl}/aiService/sse/join/${this.config.params?.stage || ''}?app-id=${this.config.params?.appId || ''}&dialog-session-id=${dialogSessionId || ''}`;
      this.eventSource = new EventSourcePolyfill(url, {
        headers: {
          Token: this.config.token || '',
          'x-session-id': this.config.token || '',
          'x-app-code': this.config.appCode || '',
        },
        withCredentials: true,
        connectionTimeout: 60000,
      });
  
      this.eventSource.onopen = (event) => {
        // 移除这里的日志，只在外部处理器中打印
        if (this.handlers.onOpen) {
          this.handlers.onOpen(event);
        }
      };

      this.eventSource.addEventListener('message', (event) => {
        try {
          const data: SSEData = JSON.parse(event.data);
          
          // 只传递原始数据，模板处理在外部进行
          if (this.handlers.onMessage) {
            this.handlers.onMessage(data);
          }
        } catch (error) {
          console.error('处理SSE消息时出错:', error);
        }
      });

      this.eventSource.onerror = (error) => {
        console.error('SSE error:', error);
        if (this.handlers.onError) {
          this.handlers.onError(error);
        }
        
        this.closeSSE();

        // 添加错误重连逻辑
        if (!this.isReconnecting.value) {
          setTimeout(() => {
            if (dialogSessionId) {
              this.reconnectSSE(dialogSessionId);
            }
          }, 3000);
        }
      };

    } catch (err) {
      console.error('初始化SSE连接失败:', err);
    }
  }

  // 重新连接SSE
  public reconnectSSE(newDialogSessionId: string): void {
    if (this.isReconnecting.value) {
      console.log('正在重连中，跳过重复重连');
      return;
    }
    this.isReconnecting.value = true;
    console.log('开始重连SSE，新的dialogSessionId:', newDialogSessionId);

    this.closeSSE();
    
    const reconnectTimeout = setTimeout(() => {
      this.initSSE(newDialogSessionId);
      setTimeout(() => {
        this.isReconnecting.value = false;
      }, 2000);
    }, 500);
    
    this.timeArr.push(reconnectTimeout);
    
    if (this.handlers.onReconnect) {
      this.handlers.onReconnect(newDialogSessionId);
    }
  }

  // 关闭SSE连接
  public closeSSE(): void {
    if (this.eventSource) {
      try {
        this.eventSource.close();
        this.eventSource = null;
      } catch (err) {
        console.warn('关闭SSE连接时出错:', err);
      }
    }

    // 清理所有定时器
    this.timeArr.forEach(timeout => {
      clearTimeout(timeout);
    });
    this.timeArr = [];
  }

  // 获取重连状态
  public getIsReconnecting(): boolean {
    return this.isReconnecting.value;
  }

  // 获取当前SSE连接状态
  public getConnectionState(): number {
    if (!this.eventSource) {
      return 0; // 未连接
    }
    return this.eventSource.readyState;
  }

  // 清理资源
  public destroy(): void {
    this.closeSSE();
    this.timeArr.forEach((item) => {
      clearTimeout(item);
    });
    this.timeArr = [];
  }
}

// 创建SSE服务实例的工厂函数
export function createSSEService(config: SSEServiceConfig, handlers: SSEHandlers = {}): SSEService {
  return new SSEService(config, handlers);
}

export default SSEService;