import { EventSourcePolyfill } from 'event-source-polyfill';
import { ref, Ref } from 'vue';

// SSE数据类型定义
export interface SSEData {
  message: any;
  status: number | string;
  type: number | string;
}

// SSE服务配置
export interface SSEServiceConfig {
  apiBaseUrl: string;
  appCode: string;
  token: string;
  params: {
    stage?: string;
    appId?: string;
  };
}

// SSE事件处理器
export interface SSEHandlers {
  onMessage?: (data: SSEData) => void;
  onError?: (error: any) => void;
  onOpen?: (event: any) => void;
  onReconnect?: (newDialogSessionId: string) => void;
  onNetworkOffline?: () => void; // 网络断开事件
  onNetworkOnline?: () => void;   // 网络恢复事件
}

// SSE服务类 - 专注于SSE连接管理
export class SSEService {
  private eventSource: EventSourcePolyfill | null = null;
  private config: SSEServiceConfig;
  private handlers: SSEHandlers;
  private isReconnecting: Ref<boolean> = ref(false);
  private timeArr: NodeJS.Timeout[] = [];
  private reconnectAttempts: number = 0;
  private maxReconnectAttempts: number = 5;
  private currentDialogSessionId: string = '';
  private connectionMonitorInterval: NodeJS.Timeout | null = null;
  
  // 心跳检测配置
  private readonly HEARTBEAT_INTERVAL = 30000; // 30秒检测一次
  private readonly RECONNECT_DELAY_BASE = 3000; // 3秒基础重连延迟
  private readonly MAX_RECONNECT_DELAY = 60000; // 最大重连延迟60秒
  private readonly MAX_RECONNECT_ATTEMPTS = 5; // 最大重连次数

  constructor(config: SSEServiceConfig, handlers: SSEHandlers = {}) {
    this.config = config;
    this.handlers = handlers;
    
    // 监听网络状态变化
    if (typeof window !== 'undefined') {
      window.addEventListener('online', this.handleNetworkOnline.bind(this));
      window.addEventListener('offline', this.handleNetworkOffline.bind(this));
    }
  }

  // 初始化SSE连接
  public initSSE(dialogSessionId: string): void {
    try {
      this.currentDialogSessionId = dialogSessionId;
      this.reconnectAttempts = 0; // 重置重连次数
      console.log('🔗 开始建立SSE连接...', '会话ID:', dialogSessionId || '空');
      
      // 即使会话ID为空，也尝试建立连接
      // 服务器可能会返回新的会话ID，或者允许空会话ID的连接
      const url = `${this.config.apiBaseUrl}/aiService/sse/join/${this.config.params?.stage || ''}?app-id=${this.config.params?.appId || ''}&dialog-session-id=${dialogSessionId || ''}`;
      this.eventSource = new EventSourcePolyfill(url, {
        headers: {
          Token: this.config.token || '',
          'x-session-id': this.config.token || '',
          'x-app-code': this.config.appCode || '',
        },
        withCredentials: true,
        connectionTimeout: 60000,
        heartbeatTimeout: 45000, // 心跳超时时间
      });
  
      this.eventSource.onopen = (event) => {
        this.reconnectAttempts = 0; // 连接成功时重置重连次数
        this.isReconnecting.value = false;
        
        console.log('✅ SSE连接已建立');
        
        // 启动心跳检测
        this.startHeartbeatCheck();
        
        if (this.handlers.onOpen) {
          this.handlers.onOpen(event);
        }
      };

      this.eventSource.addEventListener('message', (event) => {
        try {
          const data: SSEData = JSON.parse(event.data);
          
          // 重置重连次数，因为收到了有效消息
          this.reconnectAttempts = 0;
          
          // 只传递原始数据，模板处理在外部进行
          if (this.handlers.onMessage) {
            this.handlers.onMessage(data);
          }
        } catch (error) {
          console.error('处理SSE消息时出错:', error);
        }
      });

      this.eventSource.onerror = (error) => {
        console.error('❌ SSE连接错误:', error);
        if (this.handlers.onError) {
          this.handlers.onError(error);
        }
        
        this.closeSSE();

        // 检测服务器不可用情况
        const isServerUnavailable = this.isServerUnavailableError(error);
        
        if (isServerUnavailable) {
          console.error('⛔ 服务器不可用，停止重连');
          // 重置重连状态，避免后续重连
          this.reconnectAttempts = this.MAX_RECONNECT_ATTEMPTS;
          return;
        }

        // 简单重连逻辑
        if (!this.isReconnecting.value && this.reconnectAttempts < this.MAX_RECONNECT_ATTEMPTS) {
          this.reconnectAttempts++;
          const delay = this.calculateReconnectDelay();
          console.log(`🔄 SSE连接断开，将在 ${delay}ms 后尝试第 ${this.reconnectAttempts} 次重连`);
          
          setTimeout(() => {
            if (this.currentDialogSessionId) {
              this.reconnectSSE(this.currentDialogSessionId);
            }
          }, delay);
        }
      };

    } catch (err) {
      console.error('初始化SSE连接失败:', err);
    }
  }

  // 重新连接SSE
  public reconnectSSE(newDialogSessionId: string): void {
    if (this.isReconnecting.value) {
      console.log('⏳ 正在重连中，跳过重复重连');
      return;
    }
    
    if (this.reconnectAttempts >= this.maxReconnectAttempts) {
      console.error('⛔ 重连次数已达上限，停止重连');
      return;
    }
    
    this.isReconnecting.value = true;
    console.log(`🔄 开始重连SSE（第${this.reconnectAttempts}次）`, '会话ID:', newDialogSessionId || '空');

    // 只关闭SSE连接，但不停止心跳检测
    if (this.eventSource) {
      try {
        this.eventSource.close();
        this.eventSource = null;
      } catch (err) {
        console.warn('关闭SSE连接时出错:', err);
      }
    }
    
    const reconnectTimeout = setTimeout(() => {
      this.initSSE(newDialogSessionId);
      // 重连状态会在连接成功后自动重置
    }, 100);
    
    this.timeArr.push(reconnectTimeout);
    
    // 关键修复：移除这里的onReconnect回调，真正的重连成功应该在onopen事件中触发
    // 这样只有在连接真正建立成功时才会显示重连成功消息
    console.log('⏳ 重连已启动，等待连接建立...');
  }

  // 关闭SSE连接
  public closeSSE(): void {
    if (this.eventSource) {
      try {
        this.eventSource.close();
        this.eventSource = null;
      } catch (err) {
        console.warn('关闭SSE连接时出错:', err);
      }
    }

    // 停止心跳检测
    this.stopHeartbeatCheck();

    // 清理所有定时器
    this.timeArr.forEach(timeout => {
      clearTimeout(timeout);
    });
    this.timeArr = [];
    
    // 重置重连状态
    this.isReconnecting.value = false;
  }

  // 获取重连状态
  public getIsReconnecting(): boolean {
    return this.isReconnecting.value;
  }

  // 获取当前SSE连接状态
  public getConnectionState(): number {
    if (!this.eventSource) {
      return 0; // 未连接
    }
    return this.eventSource.readyState;
  }

  // 计算重连延迟（指数退避策略）
  private calculateReconnectDelay(): number {
    // 指数退避：3s, 6s, 12s, 24s, 48s
    return Math.min(this.RECONNECT_DELAY_BASE * Math.pow(2, this.reconnectAttempts - 1), this.MAX_RECONNECT_DELAY);
  }

  // 判断是否为服务器不可用错误
  private isServerUnavailableError(error: any): boolean {
    if (!error) return false;
    
    // 检查错误消息
    const errorMessage = String(error.message || '').toLowerCase();
    const errorString = String(error).toLowerCase();
    
    // 检查错误目标的状态
    const errorTarget = error.target || error;
    const readyState = errorTarget.readyState;
    
    // 服务器不可用的明确特征：
    // 1. 连接被拒绝 (net::ERR_CONNECTION_REFUSED)
    // 2. 服务器返回5xx错误
    // 3. 明确的连接失败消息
    const isConnectionRefused = errorMessage.includes('connection_refused') || 
                               errorMessage.includes('connection refused') ||
                               errorMessage.includes('failed to fetch') ||
                               errorString.includes('connection_refused') ||
                               errorString.includes('connection refused');
    
    const isServerError = errorMessage.includes('500') || 
                         errorMessage.includes('502') || 
                         errorMessage.includes('503') ||
                         errorMessage.includes('504') ||
                         errorString.includes('500') ||
                         errorString.includes('502') ||
                         errorString.includes('503') ||
                         errorString.includes('504');
    
    // 只有当明确是服务器不可用时才停止重连
    // 避免将正常的网络波动误判为服务器不可用
    const isUnavailable = isConnectionRefused || isServerError;
    
    if (isUnavailable) {
      console.log('🔍 检测到服务器不可用错误:', {
        message: errorMessage,
        string: errorString,
        readyState: readyState,
        isConnectionRefused,
        isServerError
      });
    }
    
    return isUnavailable;
  }



  // 处理网络恢复
  private handleNetworkOnline(): void {
    console.log('🌐 网络已恢复，检查SSE连接状态');
    console.log('📊 当前状态 - 重连中:', this.isReconnecting.value, '会话ID:', this.currentDialogSessionId, '重连次数:', this.reconnectAttempts);
    
    // 检查当前连接状态
    const isConnected = this.eventSource && this.eventSource.readyState === 1;
    
    if (isConnected) {
      console.log('✅ 网络恢复但SSE连接正常，无需重连');
      // 即使连接正常，也触发网络恢复回调，但只在真正需要时显示消息
      if (this.handlers.onNetworkOnline) {
        this.handlers.onNetworkOnline();
      }
      return;
    }
    
    // 如果不在重连中且连接异常，尝试重新连接
    if (!this.isReconnecting.value) {
      // 无论之前是否有会话ID，都重新创建连接
      console.log('🔄 网络恢复，创建新SSE会话');
      this.reconnectSSE('');
    } else {
      console.log('⚠️ 网络恢复但未触发重连 - 重连中:', this.isReconnecting.value, '会话ID存在:', !!this.currentDialogSessionId);
      console.log('💡 等待重连完成或心跳检测处理连接状态');
    }
    
    // 触发网络恢复回调
    if (this.handlers.onNetworkOnline) {
      this.handlers.onNetworkOnline();
    }
  }

  // 处理网络断开
  private handleNetworkOffline(): void {
    console.log('🌐 网络已断开，关闭SSE连接');
    
    // 保存当前的会话ID，保持会话连续性
    console.log('💾 当前会话ID:', this.currentDialogSessionId || '空');
    
    // 关闭SSE连接
    if (this.eventSource) {
      try {
        this.eventSource.close();
        this.eventSource = null;
      } catch (err) {
        console.warn('关闭SSE连接时出错:', err);
      }
    }
    
    // 停止心跳检测
    this.stopHeartbeatCheck();
    
    // 重置重连状态和重连次数
    this.isReconnecting.value = false;
    this.reconnectAttempts = 0;
    
    // 关键修改：保持会话ID不变，但标记网络断开状态
    // 这样网络恢复时可以继续使用同一个会话，但需要重新开始对话块
    console.log('💡 网络断开，保持会话ID，准备网络恢复时重新开始对话块');
    
    // 网络断开时启动心跳检测，以便网络恢复时能立即发现
    this.startHeartbeatCheck();
    
    // 触发网络断开回调
    if (this.handlers.onNetworkOffline) {
      this.handlers.onNetworkOffline();
    }
  }

  // 启动心跳检测
  private startHeartbeatCheck(): void {
    // 停止之前的心跳检测
    this.stopHeartbeatCheck();
    
    console.log(`💓 心跳检测已启动（${this.HEARTBEAT_INTERVAL/1000}秒间隔）`);
    
    // 使用固定间隔开始检测
    this.connectionMonitorInterval = setInterval(() => {
      this.performHeartbeatCheck();
    }, this.HEARTBEAT_INTERVAL);
  }
  
  // 执行心跳检测
  private performHeartbeatCheck(): void {
    // 检查网络状态
    const isOnline = typeof window !== 'undefined' && window.navigator.onLine;
    
    if (!isOnline) {
      console.log('💓 SSE心跳检测 - 网络已断开，等待网络恢复');
      return;
    }
    
    if (this.eventSource && this.eventSource.readyState === 1) {
      // 连接正常，记录心跳
      console.log('💓 SSE心跳检测正常 - 连接状态:', this.eventSource.readyState);
      
    } else {
      console.log('💓 SSE心跳检测 - 连接状态:', this.eventSource ? this.eventSource.readyState : '未连接');
      // 如果未连接或连接异常，但网络正常，尝试重连
      if (!this.isReconnecting.value) {
        if (this.currentDialogSessionId) {
          console.log('💔 检测到SSE连接异常，尝试重连');
          this.reconnectSSE(this.currentDialogSessionId);
        } else {
          console.log('💡 检测到网络已恢复但无会话ID，等待用户交互或网络恢复处理');
          // 不再自动触发重连，避免重复消息
          // 网络恢复事件会由网络状态监听器自动处理
        }
      }
    }
  }

  // 停止心跳检测
  private stopHeartbeatCheck(): void {
    if (this.connectionMonitorInterval) {
      clearInterval(this.connectionMonitorInterval);
      this.connectionMonitorInterval = null;
      console.log('💓 基础心跳检测已停止');
    }
  }

  // 清理资源
  public destroy(): void {
    this.closeSSE();
    this.timeArr.forEach((item) => {
      clearTimeout(item);
    });
    this.timeArr = [];
  }
}

// 创建SSE服务实例的工厂函数
export function createSSEService(config: SSEServiceConfig, handlers: SSEHandlers = {}): SSEService {
  return new SSEService(config, handlers);
}

export default SSEService;