/**
 * Markdown模板工具类
 * 用于解析和渲染Markdown内容
 */
/**
 * 处理文本中的Markdown格式（粗体、斜体、删除线、行内代码、标题等）
 * @param text 要处理的文本
 * @returns 处理后的文本
 */
const processMarkdownFormat = (text: string): string => {
  let processedText = text;
  // 处理标题（支持1-6级）
  processedText = processedText.replace(/^######\s+(.*)$/gim, '<h6>$1</h6>');
  processedText = processedText.replace(/^#####\s+(.*)$/gim, '<h5>$1</h5>');
  processedText = processedText.replace(/^####\s+(.*)$/gim, '<h4>$1</h4>');
  processedText = processedText.replace(/^###\s+(.*)$/gim, '<h3>$1</h3>');
  processedText = processedText.replace(/^##\s+(.*)$/gim, '<h2>$1</h2>');
  processedText = processedText.replace(/^#\s+(.*)$/gim, '<h1>$1</h1>');
  // 处理粗体和斜体
  processedText = processedText.replace(/\*\*(.*?)\*\*/gim, '<strong>$1</strong>');
  processedText = processedText.replace(/\*(.*?)\*/gim, '<em>$1</em>');
  processedText = processedText.replace(/__(.*?)__/gim, '<strong>$1</strong>');
  processedText = processedText.replace(/_(.*?)_/gim, '<em>$1</em>');

  // 处理删除线
  processedText = processedText.replace(/~~(.*?)~~/gim, '<del>$1</del>');

  // 处理行内代码
  processedText = processedText.replace(/`([^`]+)`/gim, '<code>$1</code>');

  return processedText;
};

/**
 * 生成完整的表格HTML
 * @param header 表头
 * @param alignments 对齐方式
 * @param dataRows 数据行
 * @returns 表格HTML
 */
const generateCompleteTable = (header: string[], alignments: string[], dataRows: string[][]): string => {
  let tableHtml = '<div class="table-container">\n';
  tableHtml += '  <table class="markdown-table" style="border-collapse: collapse; width: auto; min-width: 100%; border: 1px solid #ddd;">\n';

  // 表头
  tableHtml += '  <thead>\n    <tr>\n';
  header.forEach((cell, index) => {
    const align = alignments[index] || 'left';
    const processedCell = processMarkdownFormat(cell);
    tableHtml += `      <th style="text-align: ${align}; border: 1px solid #ddd; padding: 8px; background-color: #f5f5f5; font-weight: bold;">${processedCell}</th>\n`;
  });
  tableHtml += '    </tr>\n  </thead>\n';

  // 表体
  if (dataRows.length > 0) {
    tableHtml += '  <tbody>\n';
    dataRows.forEach(row => {
      tableHtml += '    <tr>\n';
      row.forEach((cell, index) => {
        const align = alignments[index] || 'left';
        const processedCell = processMarkdownFormat(cell);
        tableHtml += `      <td style="text-align: ${align}; border: 1px solid #ddd; padding: 8px;">${processedCell}</td>\n`;
      });
      tableHtml += '    </tr>\n';
    });
    tableHtml += '  </tbody>\n';
  }

  tableHtml += '  </table>\n';
  tableHtml += '</div>';
  return tableHtml;
};

/**
 * 增强的Markdown解析器
 * @param text Markdown文本内容
 * @returns 解析后的HTML字符串
 */
export const parseMarkdown = (text: string): string => {
  // 确保text是字符串
  if (typeof text !== 'string') {
    text = String(text || '');
  }

  // 处理引用块（必须在HTML转义之前，避免>被转义为&gt;）
  // 改进的多行引用块处理，支持连续引用行
  text = text.replace(/^>\s*(.*(?:\n>\s*.*)*)$/gim, (match, content) => {
    // 将多行引用内容合并，并用<br>分隔
    const lines = content.split(/\n>\s*/);
    const processedContent = lines.map(line => line.trim()).join('<br>');
    return `<blockquote>${processedContent}</blockquote>`;
  });

  // 处理图片（必须在任何格式处理之前，避免图片格式被破坏）
  text = text.replace(/!\[([^\]]*)\]\(([^\)]+)\)/g, (match, alt, src) => {
    const altText = alt || '图片';
    const imageId = 'img_' + Math.random().toString(36).substr(2, 9);
    return `<div class="markdown-image-container" data-image-src="${src}">
      <img src="${src}" alt="${altText}" style="cursor: pointer;" onclick="window.open('${src}', '_blank')">
      ${alt ? `<div class="image-caption">${alt}</div>` : ''}
    </div>`;
  });

  // 处理基本的Markdown格式（在表格解析之前处理，确保表格单元格中的格式也能被处理）
  text = processMarkdownFormat(text);

  // 对于完整文本，使用简单的表格处理逻辑
  // 使用正则表达式处理表格，避免复杂的流式处理
  text = text.replace(/(\|[^\n]+\|(?:\r?\n|\r))+/g, (match) => {
    const lines = match.trim().split(/\r?\n/).filter(line => line.trim() && line.includes('|'));
    if (lines.length < 2) return match;

    try {
      // 解析表头
      const headerRow = lines[0];
      const headerCells = headerRow.split('|').slice(1, -1).map(cell => cell.trim());

      // 解析对齐方式
      let alignments: string[] = [];
      let dataStartIndex = 1;

      // 查找分隔线
      for (let i = 1; i < lines.length; i++) {
        const line = lines[i];
        const cells = line.split('|').slice(1, -1).map(cell => cell.trim());
        if (cells.every(cell => /^:?-+:?$/.test(cell))) {
          alignments = cells.map(cell => {
            if (cell.startsWith(':') && cell.endsWith(':')) return 'center';
            if (cell.endsWith(':')) return 'right';
            return 'left';
          });
          dataStartIndex = i + 1;
          break;
        }
      }

      // 如果没有找到分隔线，使用默认对齐方式
      if (alignments.length === 0) {
        alignments = headerCells.map(() => 'left');
      }

      // 解析数据行
      const dataRows: string[][] = [];
      for (let i = dataStartIndex; i < lines.length; i++) {
        const row = lines[i];
        const cells = row.split('|').slice(1, -1).map(cell => cell.trim());
        dataRows.push(cells);
      }

      // 生成完整表格HTML
      return generateCompleteTable(headerCells, alignments, dataRows);
    } catch (error) {
      console.warn('表格解析失败:', error);
      return match;
    }
  });

  // 使用统一的Markdown格式处理函数处理基础格式（包括标题）
  text = processMarkdownFormat(text);

  // 处理代码块（支持语言标识）- 必须在换行处理之前
  text = text.replace(/```(\w+)?\n([\s\S]*?)```/gim, '<pre><code class="language-$1">$2</code></pre>');
  text = text.replace(/```([\s\S]*?)```/gim, '<pre><code>$1</code></pre>');

  // 处理链接（支持相对路径和绝对路径）
  text = text.replace(/\[([^\]]+)\]\(([^)]+)\)/gim, (match, text, url) => {
    // 验证URL格式
    const isValidUrl = /^(https?|ftp):\/\/[^\s/$.#?].[^\s]*$/i.test(url);
    const target = isValidUrl ? 'target="_blank" rel="noopener noreferrer"' : '';
    return `<a href="${url}" ${target}>${text}</a>`;
  });

  // 处理有序列表
  text = text.replace(/^\d+\.\s+(.*)$/gim, '<li>$1</li>');
  text = text.replace(/(<li>.*<\/li>)(?=\s*<li>)/gim, '$1');
  text = text.replace(/(<li>.*<\/li>)/gim, '<ol>$1</ol>');

  // 处理无序列表
  text = text.replace(/^[-*+]\s+(.*)$/gim, '<li>$1</li>');
  text = text.replace(/(<li>.*<\/li>)/gim, '<ul>$1</ul>');

  // 处理水平分割线
  text = text.replace(/^\s*---\s*$/gim, '<hr />');
  text = text.replace(/^\s*\*\*\*\s*$/gim, '<hr />');

  // 完全删除段落处理逻辑，避免表格被p标签包裹
  // 只处理换行，不添加p标签
  text = text.replace(/\n{3,}/g, '\n\n'); // 多个换行合并为两个
  text = text.replace(/\n\n/g, '<br><br>');
  text = text.replace(/\n/g, '<br>');

  // 清理HTML结构（移除空行）
  text = text.replace(/(<br>\s*)+/g, '<br>');

  return text;
};



/**
 * Markdown模板函数
 * @param content 要处理的Markdown内容
 * @param isStreaming 是否使用流式处理（针对SSE逐条返回的数据）
 * @returns 渲染后的HTML字符串
 */
export const markdownTemplate = (content: any, isStreaming: boolean = false): string => {
  // 类型检查和默认值处理
  if (typeof content !== 'string') {
    // 如果是对象，尝试转换为字符串
    if (content && typeof content === 'object') {
      content = JSON.stringify(content);
    } else {
      // 其他类型转换为字符串
      content = String(content || '');
    }
  }

  // 根据是否流式处理选择不同的解析方式
  const htmlContent = isStreaming ? processStreamingMarkdown(content) : parseMarkdown(content);

  // 清理HTML内容：移除不必要的br标签和空p段落
  const cleanHtml = htmlContent
    .trim()
    // 保留blockquote和code块内的换行符，移除其他不必要的br标签
    .replace(/<br\s*\/?>/gi, (match, offset, originalString) => {
      // 检查当前br标签是否在blockquote或code/pre标签内
      const beforeContent = originalString.substring(0, offset);
      const afterContent = originalString.substring(offset + match.length);
      
      // 检查前后是否有未闭合的blockquote标签
      const openBlockquotesBefore = (beforeContent.match(/<blockquote/g) || []).length;
      const closeBlockquotesBefore = (beforeContent.match(/<\/blockquote>/g) || []).length;
      const openBlockquotesAfter = (afterContent.match(/<blockquote/g) || []).length;
      const closeBlockquotesAfter = (afterContent.match(/<\/blockquote>/g) || []).length;
      
      // 检查是否在code/pre标签内
      const isInCodeBlock = beforeContent.includes('<pre>') || beforeContent.includes('<code>');
      
      const isInBlockquote = (openBlockquotesBefore - closeBlockquotesBefore) > 0;
      
      return (isInBlockquote || isInCodeBlock) ? match : '';
    })
    // 移除空的<p>段落（只包含空格或换行符）
    .replace(/<p[^>]*>\s*<\/p>/gi, '')
    // 移除只包含空格的<p>段落
    .replace(/<p[^>]*>(\s|&nbsp;)*<\/p>/gi, '')
    // 移除连续的空白段落
    .replace(/(<\/p>\s*<p[^>]*>)+/gi, '')
    // 移除开头和结尾的空白段落
    .replace(/^\s*<p[^>]*>\s*<\/p>/gi, '')
    .replace(/<p[^>]*>\s*<\/p>\s*$/gi, '')
    .trim();

  // 检查内容是否为空或只有空白
  if (!cleanHtml || cleanHtml === '<p></p>' || cleanHtml === '<p>&nbsp;</p>') {
    return ''; // 如果内容为空，返回空字符串不展示
  }

  return `<div class="message-markdown">
    <div class="markdown-content">${cleanHtml}</div>
  </div>`;
};

/**
 * 简化的Markdown生成函数（兼容旧版本）
 * @param content Markdown内容
 * @param isStreaming 是否使用流式处理（可选参数）
 * @returns 渲染后的HTML字符串
 */
export const markdown = (content: any, isStreaming: boolean = false): string => {
  return markdownTemplate(content, isStreaming);
};

/**
 * 检查内容块是否是markdown块
 * @param contentBlock 内容块对象
 * @returns 是否是markdown块
 */
export const isMarkdownBlock = (contentBlock: any): boolean => {
  return contentBlock && contentBlock.content && contentBlock.content.includes('message-markdown');
};

/**
 * 检查文本是否是表格行
 * @param text 要检查的文本
 * @returns 是否是表格行
 */
export const isTableRow = (text: string): boolean => {
  if (!text || typeof text !== 'string') {
    return false;
  }
  const trimmedText = text.trim();
  return trimmedText.startsWith('|') && trimmedText.endsWith('|');
};

/**
 * 检查文本是否是表格分隔线
 * @param text 要检查的文本
 * @returns 是否是表格分隔线
 */
export const isTableSeparator = (text: string): boolean => {
  if (!text || typeof text !== 'string') {
    return false;
  }
  const trimmedText = text.trim();
  if (!trimmedText.startsWith('|') || !trimmedText.endsWith('|')) {
    return false;
  }

  const cells = trimmedText.split('|').slice(1, -1).map(cell => cell.trim());
  return cells.every(cell => /^:?-+:?$/.test(cell));
};

/**
 * 检查文本是否是表格的开始（表头）
 * @param text 要检查的文本
 * @returns 是否是表格开始
 */
export const isTableStart = (text: string): boolean => {
  if (!text || typeof text !== 'string') {
    return false;
  }
  const trimmedText = text.trim();
  return trimmedText.startsWith('|') && trimmedText.endsWith('|') && !isTableSeparator(trimmedText);
};

/**
 * 流式表格处理器
 * 用于处理SSE逐条返回的表格数据
 */
class StreamingTableProcessor {
  private tableData: string[] = [];
  private isInTable: boolean = false;

  /**
   * 处理新的文本内容
   * @param text 新的文本内容
   * @returns 处理结果：如果是表格行返回null，表格完成时返回完整表格HTML
   */
  public processText(text: string): string | null {
    if (!text || typeof text !== 'string') {
      return null;
    }

    const trimmedText = text.trim();

    // 检查是否是表格行
    if (isTableRow(trimmedText)) {
      if (!this.isInTable) {
// 开始新的表格
        this.isInTable = true;
        this.tableData = [trimmedText];
        return null; // 表格开始，不立即渲染
      } else {
        // 继续当前表格，收集所有表格行
        this.tableData.push(trimmedText);

        // 不立即生成表格，继续收集直到遇到非表格行
        return null; // 继续收集表格行
      }
    } else {
      // 非表格行
      if (this.isInTable && this.tableData.length > 0) {
        // 表格结束，处理缓存的表格数据
        let result: string | null = null;

        // 检查表格是否至少包含表头和数据行（至少2行）
        if (this.tableData.length >= 2) {
          // 表格完整，生成完整表格
          result = this.generateCompleteTable();
        } else {
          // 表格不完整，显示加载提示
          result = '<div class="table-loading" style="border: 1px solid #ddd; padding: 10px; background-color: #f9f9f9; margin: 10px 0;">表格数据加载中...</div>';
        }

        this.reset();
        return result;
      }

      // 普通文本，直接返回
      return text;
    }
  }

  /**
   * 生成完整表格HTML
   * @returns 表格HTML
   */
  private generateCompleteTable(): string {
    if (this.tableData.length < 2) {
      return '';
    }

    try {
      // 解析表头
      const headerRow = this.tableData[0];
      const headerCells = headerRow.split('|').slice(1, -1).map(cell => {
        // 对表头单元格进行Markdown格式处理
        return processMarkdownFormat(cell.trim());
      });

      // 解析对齐方式（从第二行）
      let alignments: string[] = [];
      let dataStartIndex = 1;

      // 查找分隔线
      for (let i = 1; i < this.tableData.length; i++) {
        if (isTableSeparator(this.tableData[i])) {
          const alignRow = this.tableData[i];
          const alignCells = alignRow.split('|').slice(1, -1).map(cell => cell.trim());
          alignments = alignCells.map(cell => {
            if (cell.startsWith(':') && cell.endsWith(':')) return 'center';
            if (cell.endsWith(':')) return 'right';
            return 'left';
          });
          dataStartIndex = i + 1;
          break;
        }
      }

      // 如果没有找到分隔线，使用默认对齐方式，数据从第二行开始
      if (alignments.length === 0) {
        alignments = headerCells.map(() => 'left');
        dataStartIndex = 1; // 对于无分隔线表格，数据从第二行开始
      }

      // 解析数据行
      const dataRows: string[][] = [];
      for (let i = dataStartIndex; i < this.tableData.length; i++) {
        const row = this.tableData[i];
        const cells = row.split('|').slice(1, -1).map(cell => {
          // 对数据单元格进行Markdown格式处理
          return processMarkdownFormat(cell.trim());
        });
        dataRows.push(cells);
      }

      // 生成完整表格HTML
      return generateCompleteTable(headerCells, alignments, dataRows);
    } catch (error) {
      console.warn('表格生成失败:', error);
      return '<div class="table-error" style="border: 1px solid #ff6b6b; padding: 10px; background-color: #ffeaea; margin: 10px 0;">表格渲染失败</div>';
    }
  }

  /**
   * 重置表格处理器状态
   */
  private reset(): void {
    this.tableData = [];
    this.isInTable = false;
  }

  /**
   * 获取当前表格状态
   */
  public getTableState(): { isInTable: boolean; tableData: string[] } {
    return {
      isInTable: this.isInTable,
      tableData: [...this.tableData]
    };
  }
}

// 创建全局表格处理器实例
const streamingTableProcessor = new StreamingTableProcessor();

/**
 * 检查文本是否是列表项（有序或无序）
 * @param text 要检查的文本
 * @returns 是否是列表项
 */
export const isListItem = (text: string): boolean => {
  if (!text || typeof text !== 'string') {
    return false;
  }
  const trimmedText = text.trim();
  // 匹配有序列表（1. 2. 等）或无序列表（- * +）
  return /^\d+\.\s+/.test(trimmedText) || /^[-*+]\s+/.test(trimmedText);
};

/**
 * 检查文本是否是有序列表项
 * @param text 要检查的文本
 * @returns 是否是有序列表项
 */
export const isOrderedListItem = (text: string): boolean => {
  if (!text || typeof text !== 'string') {
    return false;
  }
  const trimmedText = text.trim();
  return /^\d+\.\s+/.test(trimmedText);
};

/**
 * 检查文本是否是无序列表项
 * @param text 要检查的文本
 * @returns 是否是无序列表项
 */
export const isUnorderedListItem = (text: string): boolean => {
  if (!text || typeof text !== 'string') {
    return false;
  }
  const trimmedText = text.trim();
  return /^[-*+]\s+/.test(trimmedText);
};

/**
 * 流式列表处理器
 * 用于处理SSE逐条返回的列表数据
 */
class StreamingListProcessor {
  private listItems: string[] = [];
  private isInList: boolean = false;
  private listType: 'ordered' | 'unordered' | null = null;
  private orderedListStartNumber: number = 1; // 记录有序列表的起始序号

  /**
   * 处理新的文本内容
   * @param text 新的文本内容
   * @returns 处理结果：如果是列表项返回null，列表完成时返回完整列表HTML
   */
  public processText(text: string): string | null {
    if (!text || typeof text !== 'string') {
      return null;
    }

    const trimmedText = text.trim();

    // 检查是否是列表项
    if (isListItem(trimmedText)) {
      const isOrdered = isOrderedListItem(trimmedText);
      const currentListType = isOrdered ? 'ordered' : 'unordered';

      if (!this.isInList) {
        // 开始新的列表
        this.isInList = true;
        this.listType = currentListType;
        this.listItems = [trimmedText];
        // 如果是有序列表，记录起始序号
        if (isOrdered) {
          const match = trimmedText.match(/^(\d+)\.\s+/);
          this.orderedListStartNumber = match ? parseInt(match[1]) : 1;
        }
        return null; // 列表开始，不立即渲染
      } else if (this.listType === currentListType) {
        // 继续当前列表，收集所有列表项
        this.listItems.push(trimmedText);
        return null; // 继续收集列表项
      } else {
        // 列表类型改变，结束当前列表并开始新的列表
        const result = this.generateCompleteList();
        this.isInList = true;
        this.listType = currentListType;
        this.listItems = [trimmedText];
        // 如果是有序列表，记录起始序号
        if (isOrdered) {
          const match = trimmedText.match(/^(\d+)\.\s+/);
          this.orderedListStartNumber = match ? parseInt(match[1]) : 1;
        }
        return result;
      }
    } else {
      // 非列表项
      if (this.isInList && this.listItems.length > 0) {
        // 列表结束，处理缓存的列表数据
        const result = this.generateCompleteList();
        this.reset();
        return result;
      }

      // 普通文本，直接返回
      return text;
    }
  }

  /**
   * 生成完整列表HTML
   * @returns 列表HTML
   */
  private generateCompleteList(): string {
    if (this.listItems.length === 0) {
      return '';
    }

    try {
      // 处理每个列表项的内容
      const processedItems = this.listItems.map((item, index) => {
        if (this.listType === 'ordered') {
          // 有序列表：移除数字标记，保留内容，但保持正确的序号
          const content = item.replace(/^\d+\.\s+/, '').trim();
          const processedContent = processMarkdownFormat(content);
          return `<li value="${this.orderedListStartNumber + index}">${processedContent}</li>`;
        } else {
          // 无序列表：移除标记，保留内容
          const content = item.replace(/^[-*+]\s+/, '').trim();
          const processedContent = processMarkdownFormat(content);
          return `<li>${processedContent}</li>`;
        }
      });

      // 根据列表类型生成完整列表
      const listTag = this.listType === 'ordered' ? 'ol' : 'ul';
      return `<${listTag}>${processedItems.join('')}</${listTag}>`;
    } catch (error) {
      console.warn('列表生成失败:', error);
      return '<div class="list-error" style="border: 1px solid #ff6b6b; padding: 10px; background-color: #ffeaea; margin: 10px 0;">列表渲染失败</div>';
    }
  }

  /**
   * 重置列表处理器状态
   */
  private reset(): void {
    this.listItems = [];
    this.isInList = false;
    this.listType = null;
    this.orderedListStartNumber = 1;
  }

  /**
   * 获取当前列表状态
   */
  public getListState(): { isInList: boolean; listType: 'ordered' | 'unordered' | null; listItems: string[]; orderedListStartNumber: number } {
    return {
      isInList: this.isInList,
      listType: this.listType,
      listItems: [...this.listItems],
      orderedListStartNumber: this.orderedListStartNumber
    };
  }
}

// 创建全局列表处理器实例
const streamingListProcessor = new StreamingListProcessor();

/**
 * 处理流式Markdown文本（专门用于SSE逐条返回的数据）
 * @param text 新的文本内容
 * @returns 处理后的HTML内容
 */
export const processStreamingMarkdown = (text: string): string => {
  // 清理危险的HTML标签，防止XSS攻击，但保留图片和必要的容器标签
  text = text.replace(/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/gi, '');
  // 移除其他危险标签，但保留img、div、span等安全标签
  text = text.replace(/<(?!\/?(img|div|span|p|br|ol|ul|li|table|tr|td|th|blockquote|pre|code|strong|em|a|h[1-6])(\s+[^>]*)?>)[^>]*>/gi, '');

  // 使用流式列表处理器处理文本
  const listResult = streamingListProcessor.processText(text);

  if (listResult === null) {
    // 列表项被收集，不立即渲染
    return '';
  }

  // 如果列表处理器返回了完整列表，需要检查是否有剩余内容需要处理
  if (listResult && (listResult.includes('<ol>') || listResult.includes('<ul>'))) {
    // 检查是否有非列表内容需要处理
    const remainingText = text.trim();
    if (remainingText && !isListItem(remainingText)) {
      // 有剩余的非列表内容（如图片），需要继续处理
      const remainingResult = parseMarkdown(remainingText);
      return listResult + remainingResult;
    }
    return listResult;
  }

  // 使用流式表格处理器处理文本
  const tableResult = streamingTableProcessor.processText(text);

  if (tableResult === null) {
    // 表格行被收集，不立即渲染
    return '';
  }

  // 如果表格处理器返回了完整表格，需要确保表格内容中的Markdown格式也被处理
  if (tableResult.includes('<table')) {
    // 对表格内容进行Markdown格式处理
    return processMarkdownFormat(tableResult);
  }

  // 否则，处理其他Markdown语法
  return parseMarkdown(tableResult || text);
};

/**
 * 检查最后一个contentBlock是否是markdown块
 * @param contentBlocks 内容块数组
 * @returns 最后一个块是否是markdown块
 */
export const isLastBlockMarkdown = (contentBlocks: any[]): boolean => {
  if (!contentBlocks || contentBlocks.length === 0) {
    return false;
  }
  const lastBlock = contentBlocks[contentBlocks.length - 1];
  return isMarkdownBlock(lastBlock);
};

/**
 * 获取最后一个markdown块的索引
 * @param contentBlocks 内容块数组
 * @returns 最后一个markdown块的索引，如果没有找到返回-1
 */
export const getLastMarkdownBlockIndex = (contentBlocks: any[]): number => {
  if (!contentBlocks || contentBlocks.length === 0) {
    return -1;
  }
  for (let i = contentBlocks.length - 1; i >= 0; i--) {
    if (isMarkdownBlock(contentBlocks[i])) {
      return i;
    }
  }
  return -1;
};

/**
 * 合并markdown内容
 * @param existingContent 现有的markdown内容
 * @param newContent 新的markdown内容
 * @returns 合并后的markdown内容
 */
export const mergeMarkdownContent = (existingContent: string, newContent: string): string => {
  // 从现有的markdown内容中提取内部内容
  const existingInnerContent = existingContent.replace(/<div class="message-markdown">\s*<div class="markdown-content">([\s\S]*?)<\/div>\s*<\/div>/, '$1');

  // 从新的markdown内容中提取内部内容
  const newInnerContent = newContent.replace(/<div class="message-markdown">\s*<div class="markdown-content">([\s\S]*?)<\/div>\s*<\/div>/, '$1');

  // 合并内容并重新包裹
  const mergedInnerContent = existingInnerContent + newInnerContent;

  return `<div class="message-markdown">
    <div class="markdown-content">${mergedInnerContent}</div>
  </div>`;
};

export default {
  parseMarkdown, // 解析Markdown文本为HTML
  markdownTemplate, // 应用Markdown模板
  markdown, // 完整的Markdown处理函数
  isMarkdownBlock, // 检查是否为Markdown块
  isLastBlockMarkdown, // 检查最后一个块是否为Markdown
  getLastMarkdownBlockIndex, // 获取最后一个Markdown块的索引
  mergeMarkdownContent, // 合并Markdown内容
  isListItem, // 检查是否为列表项
  isOrderedListItem, // 检查是否为有序列表项
  isUnorderedListItem // 检查是否为无序列表项
};