import dayjs from 'dayjs';
import { tableTemplate } from './tableTemplate';
import { markdownTemplate, isLastBlockMarkdown, getLastMarkdownBlockIndex, mergeMarkdownContent } from './markdownTemplate';

// 内容模板类型定义
  export interface ContentTemplates {
    text: (content: string) => string;
    thinking: (content: string) => string;
    error: (content: string) => string;
    table: (tableData: any) => string;
    markdown: (content: any, isStreaming?: boolean) => string;
    option: (optionData: any) => string;
    iframe: (iframeData: any) => string;
  }

// 消息块类型定义
export interface MessageBlock {
  content: string;
  thinkContent?: string;
  hasThinkBox: boolean;
  thinkBoxExpanded: boolean;
  chartData?: any;
  chartType?: number | string;
  audioData?: {
    src: string;
    durationTime: string;
  };
}

// 消息类型定义
export interface Message {
  messageType: 'received' | 'sent';
  avatar: string;
  recordId: string;
  promptTokens: number;
  completionTokens: number;
  totalTokens: number;
  date: string;
  customClass?: string;
  contentBlocks: MessageBlock[];
}

// SSE数据类型定义
export interface SSEData {
  message: any;
  status: number | string;
  type: number | string;
}

// 模板处理结果
export interface TemplateProcessResult {
  updatedResponse: Message | null;
  updatedIsThinking: boolean;
  updatedBlockIndex: number;
  recordId: string;
  promptTokens: number;
  completionTokens: number;
  totalTokens: number;
  newDialogSessionId: string;
}

// 内容模板服务类
export class ContentTemplateService {
  private templates: ContentTemplates;

  constructor() {
    this.templates = this.createTemplates();
  }

  // 创建内容模板生成器
  private createTemplates(): ContentTemplates {
    return {
      // 普通文本
      text: (content: string) => {
        return `<div class="message-text">${content}</div>`;
      },
      
      // 思考过程
      thinking: (content: string) => {
        const formattedContent = content
          .split('\n')
          .map((line) => `<div class="think-line">${line}</div>`)
          .join('');
        return `<div class="think-content">${formattedContent}</div>`;
      },
      
      // 错误信息
      error: (content: string) => {
        return `<div class="message-error">${content}</div>`;
      },
      
      // 表格模板 - 使用独立的表格模板工具
      table: (tableData: any) => {
        return tableTemplate(tableData);
      },
      
      // Markdown模板 - 使用独立的markdown模板工具
      markdown: (content: any, isStreaming: boolean = false) => {
        return markdownTemplate(content, isStreaming);
      },
      
      // 选项数据模板 - 纯渲染，不允许点击
      option: (optionData: any) => {
        const { tips, options } = optionData;

        // 生成选项列表HTML - 序号和标题直接放在一行
        const optionsHtml = options?.map((item: any, index: number) => {
          const { title, url } = item;
          return `
            <div class="option-item">
              <div class="option-content">
                <span class="option-number-title">${index + 1}. ${title}</span>
                <span class="option-url">(${url})</span>
              </div>
            </div>
          `;
        }).join('') || '';

        return `
          <div class="message-options">
            ${tips ? `<div class="options-tips">${tips}</div>` : ''}
            <div class="options-list">
              ${optionsHtml}
            </div>
          </div>
        `;
      },
      
      // 简化的iframe模板 - 移除全屏功能，设置宽高100%固定
      iframe: (iframeData: any) => {
        const { tips, title, url } = iframeData || {};
        console.log('iframeData', iframeData);
        return `<div class="message-iframe iframe-loading">
          <!-- 加载状态 -->
          <div class="iframe-loading">
            <div class="loading-spinner"></div>
            <div class="loading-text">正在加载内容...</div>
            <div class="loading-progress">
              <div class="progress-bar"></div>
            </div>
          </div>
          
          <!-- iframe容器 -->
          <div class="iframe-tips">${tips || ''}</div>
          <div class="iframe-title">${title || ''}</div>
          <iframe 
            src="${url}" 
            width="100%" 
            height="100%"
            frameborder="0"
            sandbox="allow-scripts allow-same-origin allow-forms allow-popups allow-popups-to-escape-sandbox"
            scrolling="no"
            style="overflow: hidden;"
            onload="this.parentElement.classList.add('iframe-loaded'); this.parentElement.classList.remove('iframe-loading');"
            onerror="console.error('iframe加载失败:', this.src)"
          ></iframe>
        </div>`;
      }
    };
  }

  // 获取模板对象
  public getTemplates(): ContentTemplates {
    return this.templates;
  }

  // 公共模板生成方法
  public generateTextTemplate(content: string): string {
    return this.templates.text(content);
  }

  public generateThinkingTemplate(content: string): string {
    return this.templates.thinking(content);
  }

  public generateErrorTemplate(content: string): string {
    return this.templates.error(content);
  }



  // 处理SSE消息的核心方法
  public processSSEMessage(
    data: SSEData,
    currentResponse: Message | null,
    isThinking: boolean,
    currentBlockIndex: number,
    isHistoryData = false,
    templateTools?: {
      tableTemplate: (tableData: any) => string;
      markdownTemplate: (content: any, isStreaming?: boolean) => string;
      isLastBlockMarkdown: (blocks: MessageBlock[]) => boolean;
      getLastMarkdownBlockIndex: (blocks: MessageBlock[]) => number;
      mergeMarkdownContent: (existing: string, newContent: string) => string;
    }
  ): TemplateProcessResult {
    let messageContent = data.message || '';
    const contentStatus = data.status;
    const contentType = data.type;
    
    let updatedResponse = currentResponse;
    let updatedIsThinking = isThinking;
    let updatedBlockIndex = currentBlockIndex;
    let recordId = '';
    let promptTokens = 0;
    let completionTokens = 0;
    let totalTokens = 0;
    let newDialogSessionId = '';

    // 根据是否为历史数据设置默认展开状态
    const defaultThinkBoxExpanded = !isHistoryData;

    switch (contentStatus) {
      case -1: // 错误信息
        if (updatedResponse) {
          updatedResponse.contentBlocks.push({
            content: this.templates.error(messageContent || '出错了~~'),
            hasThinkBox: false,
            thinkContent: '',
            thinkBoxExpanded: false,
          });
        }
        break;
        
      case 3: // 图表数据
        if (updatedResponse) {
          switch (contentType) {
            case 2: // 表格数据
              const { rows } = messageContent;
              // 表格数据处理
              updatedResponse.contentBlocks.push({
                content: templateTools?.tableTemplate ? templateTools.tableTemplate(rows) : this.templates.table(rows),
                hasThinkBox: false,
                thinkContent: '',
                thinkBoxExpanded: false,
              });
              // 图表数据处理
              updatedResponse.contentBlocks.push({
                content: '',
                hasThinkBox: false,
                thinkContent: '',
                thinkBoxExpanded: false,
                chartData: messageContent,
                chartType: 3,
              });
              break;
              
            case 3: // 选项数据
              const { tips, options } = messageContent;
              if (options?.length) {
                if (options?.length === 1) {
                  // 走iframe
                  updatedResponse.contentBlocks.push({
                    content: this.templates.iframe({
                      ...options[0],
                      tips: tips || '',
                    }),
                    hasThinkBox: false,
                    thinkContent: '',
                    thinkBoxExpanded: false,
                  });
                } else {
                  updatedResponse.contentBlocks.push({
                    content: this.templates.option(messageContent),
                    hasThinkBox: false,
                    thinkContent: '',
                    thinkBoxExpanded: false,
                  });
                }
              }
              break;
              
            case 4: // MD格式
              if (updatedResponse) {
                // 对于SSE流式数据，使用流式处理
                const markdownContent = templateTools?.markdownTemplate ? 
                  templateTools.markdownTemplate(messageContent || '', true) : 
                  this.templates.markdown(messageContent || '', true);

                // 检查最后一个块是否是markdown块
                const isLastMarkdown = templateTools?.isLastBlockMarkdown ?
                  templateTools.isLastBlockMarkdown(updatedResponse.contentBlocks) :
                  isLastBlockMarkdown(updatedResponse.contentBlocks);
                  
                if (isLastMarkdown) {
                  // 合并到现有的markdown块
                  const lastMarkdownIndex = templateTools?.getLastMarkdownBlockIndex ?
                    templateTools.getLastMarkdownBlockIndex(updatedResponse.contentBlocks) :
                    getLastMarkdownBlockIndex(updatedResponse.contentBlocks);
                  
                  if (lastMarkdownIndex !== -1) {
                    updatedResponse.contentBlocks[lastMarkdownIndex].content =
                      templateTools?.mergeMarkdownContent ?
                      templateTools.mergeMarkdownContent(
                        updatedResponse.contentBlocks[lastMarkdownIndex].content,
                        markdownContent
                      ) :
                      mergeMarkdownContent(
                        updatedResponse.contentBlocks[lastMarkdownIndex].content,
                        markdownContent
                      );
                  }
                } else {
                  // 创建新的markdown块
                  updatedResponse.contentBlocks.push({
                    content: markdownContent,
                    hasThinkBox: false,
                    thinkContent: '',
                    thinkBoxExpanded: false,
                  });
                }
              }
              break;
              
            default: // 默认处理
              updatedResponse.contentBlocks.push({
                content: this.templates.text(messageContent || ''),
                hasThinkBox: false,
                thinkContent: '',
                thinkBoxExpanded: false,
              });
              break;
          }
        }
        break;
        
      case 10: // 思考开始
        updatedIsThinking = true;
        if (updatedBlockIndex === -1 && updatedResponse) {
          updatedBlockIndex = updatedResponse.contentBlocks.length;
          updatedResponse.contentBlocks.push({
            content: '',
            thinkContent: `${messageContent}`,
            hasThinkBox: true,
            thinkBoxExpanded: defaultThinkBoxExpanded,
          });
        } else if (updatedResponse && updatedResponse.contentBlocks[updatedBlockIndex]) {
          updatedResponse.contentBlocks[updatedBlockIndex].thinkContent += ``;
          updatedResponse.contentBlocks[updatedBlockIndex].hasThinkBox = true;
          updatedResponse.contentBlocks[updatedBlockIndex].thinkBoxExpanded = defaultThinkBoxExpanded;
        }
        break;

      case 11: // 思考结束
        if (updatedResponse && updatedBlockIndex !== -1 && updatedResponse.contentBlocks[updatedBlockIndex]) {
          updatedResponse.contentBlocks[updatedBlockIndex].thinkContent += ``;
          updatedResponse.contentBlocks[updatedBlockIndex].hasThinkBox = true;
          updatedResponse.contentBlocks[updatedBlockIndex].thinkBoxExpanded = defaultThinkBoxExpanded;
        }
        updatedIsThinking = false;
        break;

      case 20: // 初始连接回传信息
        if (updatedResponse) {
          updatedResponse.contentBlocks.push({
            content: this.templates.text(messageContent?.words || ''),
            hasThinkBox: false,
            thinkContent: '',
            thinkBoxExpanded: false,
          });
        }
        newDialogSessionId = messageContent?.dialogSessionId || '';
        break;
        
      case 21: // 重连成功正回传信息
        newDialogSessionId = messageContent?.dialogSessionId || '';
        break;

      case 29: // 当前会话结束
        recordId = messageContent?.recordId || '';
        promptTokens = messageContent?.promptTokens || 0;
        completionTokens = messageContent?.completionTokens || 0;
        totalTokens = messageContent?.totalTokens || 0;
        // 只有实时对话才在29时折叠思考框，历史数据不受影响
        if (!isHistoryData && updatedResponse && updatedResponse.contentBlocks.length > 0) {
          updatedResponse.contentBlocks.forEach((block) => {
            if (block.hasThinkBox) {
              block.thinkBoxExpanded = false;
            }
          });
        }
        updatedIsThinking = false;
        updatedBlockIndex = -1;
        break;

      default: // 普通内容
        if (updatedIsThinking && updatedResponse) {
          if (updatedBlockIndex !== -1 && updatedResponse.contentBlocks[updatedBlockIndex]) {
            updatedResponse.contentBlocks[updatedBlockIndex].thinkContent += `\n${messageContent}`;
            updatedResponse.contentBlocks[updatedBlockIndex].hasThinkBox = true;
            updatedResponse.contentBlocks[updatedBlockIndex].thinkBoxExpanded = defaultThinkBoxExpanded;
          }
        } else if (updatedResponse) {
          updatedBlockIndex = updatedResponse.contentBlocks.length;
          updatedResponse.contentBlocks.push({
            content: this.templates.text(messageContent),
            hasThinkBox: false,
            thinkContent: '',
            thinkBoxExpanded: false,
          });
        }
        break;
    }

    return {
      updatedResponse,
      updatedIsThinking,
      updatedBlockIndex,
      recordId,
      promptTokens,
      completionTokens,
      totalTokens,
      newDialogSessionId,
    };
  }

  // 处理历史记录数据
  /**
   * 处理历史记录数据，将其转换为可渲染的消息格式
   * @param dataArray 包含历史记录数据的数组
   * @param templateTools 包含模板工具函数的对象（可选）
   * @returns 转换后的消息数组
   */
  public processHistoryData(
    dataArray: any[],
    templateTools?: {
      tableTemplate: (tableData: any) => string;
      markdownTemplate: (content: any, isStreaming?: boolean) => string;
      isLastBlockMarkdown: (blocks: MessageBlock[]) => boolean;
      getLastMarkdownBlockIndex: (blocks: MessageBlock[]) => number;
      mergeMarkdownContent: (existing: string, newContent: string) => string;
    }
  ): Message[] {
    const result: Message[] = [];
    
    dataArray.forEach((data) => {
      let date = dayjs(data.startTime).format('YYYY-MM-DD HH:mm:ss');
      
      // 处理问题消息
      if (data.question || data.audioPath) {
        // 创建基础消息结构
        const message = {
          messageType: 'sent' as const,
          avatar: '我',
          recordId: '',
          promptTokens: 0,
          completionTokens: 0,
          totalTokens: 0,
          date,
          contentBlocks: [] as MessageBlock[]
        };

        // 使用switch语句处理不同类型的消息
        switch (true) {
          case !!data.audioPath:
            // 音频消息
            message.contentBlocks.push({
              audioData: {
                src: data.audioPath,
                durationTime: data.audioTime || '0"'
              },
              content: '',
              thinkContent: '',
              hasThinkBox: false,
              thinkBoxExpanded: false,
            });
            break;
            
          case !!data.question:
            // 文本消息
            message.contentBlocks.push({
              content: this.templates.text(data.question),
              thinkContent: '',
              hasThinkBox: false,
              thinkBoxExpanded: false,
            });
            break;
            
          default:
            // 其他类型的消息（未来扩展）
            break;
        }

        result.push(message);
      }
      
      // 处理AI回答消息
      if (data.answerInfoList && Array.isArray(data.answerInfoList)) {
        const aiMessage: Message = {
          messageType: 'received',
          avatar: 'AI',
          recordId: '',
          promptTokens: 0,
          completionTokens: 0,
          totalTokens: 0,
          contentBlocks: [],
          date,
        };

        let currentThinkingMode = false;
        let currentBlockIdx = -1;

        // 历史数据处理，isHistoryData设为true，思考框折叠
        data.answerInfoList.forEach((answer) => {
          const sseData: SSEData = {
            message: answer.message || '',
            status: answer.status || 0,
            type: answer.type || '',
          };

          const processResult = this.processSSEMessage(
            sseData,
            aiMessage,
            currentThinkingMode,
            currentBlockIdx,
            true,
            templateTools
          );

          currentThinkingMode = processResult.updatedIsThinking;
          currentBlockIdx = processResult.updatedBlockIndex;
          aiMessage.recordId = processResult.recordId;
          aiMessage.promptTokens = processResult.promptTokens;
          aiMessage.completionTokens = processResult.completionTokens;
          aiMessage.totalTokens = processResult.totalTokens;
        });

        if (aiMessage.contentBlocks.length > 0) {
          result.push(aiMessage);
        }
      }
    });

    return result;
  }
}

// 创建内容模板服务实例的工厂函数
export function createContentTemplateService(): ContentTemplateService {
  return new ContentTemplateService();
}

export default ContentTemplateService;