// 企业微信登录工具类 - 简化版本（直接从URL获取code）
class WeChatLogin {
  constructor() {
    this.corpId = 'ww88c20288b24a8e33' // 企业微信CorpID
    this.agentId = '1000002' // 应用AgentId
  }

  // 检查是否为企业微信浏览器
  checkWeChatWorkBrowser() {
    const ua = navigator.userAgent.toLowerCase()
    return ua.indexOf('wxwork') !== -1
  }

  // 从URL参数获取code
  getCodeFromURL() {
    const urlParams = new URLSearchParams(window.location.search)
    return urlParams.get('code')
  }

  // 企业微信静默登录 - 核心方法
  async silentLogin() {
    try {
      // 直接从URL获取code
      const code = this.getCodeFromURL() || ""

      if (!code) {
        console.error('未找到code参数，请确保在企业微信内部应用中访问')
        return {
          isLoggedIn: false,
          message: '未找到code参数，请确保在企业微信内部应用中访问'
        }
      }

      // 获取用户信息
      const userInfo = await this.getUserInfo(code)
      
      return {
        isLoggedIn: true,
        code: code,
        userInfo: userInfo
      }
    } catch (error) {
      console.error('登录失败:', error.message)
      return {
        isLoggedIn: false,
        message: error.message || '登录失败'
      }
    }
  }

  // 获取用户信息 - 先执行auth接口，再执行模型信息接口
  async getUserInfo(code) {
    try {
      // 第一步：执行auth接口获取用户信息和token
      const authResponse = await fetch('/pedapi/pedService/wxcp/auth', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          authCode: code
        })
      })

      if (!authResponse.ok) {
        throw new Error(`Auth接口HTTP错误! status: ${authResponse.status}`)
      }

      const authResult = await authResponse.json()
      
      if (authResult.code !== 0) {
        throw new Error(authResult.message || '获取用户信息失败')
      }

      // 第二步：使用auth接口返回的token调用模型信息接口
      const token = authResult.data.extMap?.sessionId || ''
      if (!token) {
        throw new Error('Auth接口未返回token')
      }

      const modelResponse = await fetch('/agentService/apps/getAppId', {
        method: 'GET',
        headers: {
          'Token': token || '',
          'x-session-id': token || '',
          'Content-Type': 'application/json',
        }
      })

      if (!modelResponse.ok) {
        throw new Error(`模型信息接口HTTP错误! status: ${modelResponse.status}`)
      }

      const modelResult = await modelResponse.json()
      
      if (modelResult.code !== 0) {
        throw new Error(modelResult.message || '获取模型信息失败')
      }

      // 合并用户信息和模型信息
      const userData = {
        ...authResult.data,
        appId: modelResult.data.appId || ''
      }

      // 保存用户信息
      localStorage.setItem('wechat_user', JSON.stringify(userData))
      return userData
    } catch (error) {
      throw new Error(`获取用户信息失败: ${error.message}`)
    }
  }

  // 检查登录状态
  checkLoginStatus() {
    const userData = localStorage.getItem('wechat_user')
    if (userData) {
      try {
        return {
          isLoggedIn: true,
          userInfo: JSON.parse(userData)
        }
      } catch (error) {
        localStorage.removeItem('wechat_user')
      }
    }
    return { isLoggedIn: false }
  }

  // 退出登录
  logout() {
    localStorage.removeItem('wechat_user')
  }

  // 获取当前用户
  getCurrentUser() {
    const status = this.checkLoginStatus()
    return status.isLoggedIn ? status.userInfo : null
  }
}

export default new WeChatLogin()