import { EventSourcePolyfill } from 'event-source-polyfill';
import { ref, Ref } from 'vue';

// SSE数据类型定义
export interface SSEData {
  message: any;
  status: number | string;
  type: number | string;
}

// SSE服务配置
export interface SSEServiceConfig {
  apiBaseUrl: string;
  appCode: string;
  token: string;
  params: {
    stage?: string;
    appId?: string;
  };
}

// SSE事件处理器
export interface SSEHandlers {
  onMessage?: (data: SSEData) => void;
  onError?: (error: any) => void;
  onOpen?: (event: any) => void;
  onReconnect?: (newDialogSessionId: string) => void;
}

// SSE服务类 - 专注于SSE连接管理
export class SSEService {
  private eventSource: EventSourcePolyfill | null = null;
  private config: SSEServiceConfig;
  private handlers: SSEHandlers;
  private isReconnecting: Ref<boolean> = ref(false);
  private timeArr: NodeJS.Timeout[] = [];
  
  // keepalive相关状态
  private hasReceivedKeepalive: boolean = false; // 标记是否收到过keepalive事件

  // 页面刷新监听器相关
  private beforeUnloadHandler: (() => void) | null = null;
  private isPageRefreshing: boolean = false;

  constructor(config: SSEServiceConfig, handlers: SSEHandlers = {}) {
    this.config = config;
    this.handlers = handlers;
    this.setupPageRefreshListener();
  }

  // 初始化SSE连接
  public initSSE(dialogSessionId: string): void {
    try {
      const url = `${import.meta.env.VITE_SSE_PATH}/sse/join/${this.config.params?.stage || ''}?app-id=${this.config.params?.appId || ''}&dialog-session-id=${dialogSessionId || ''}`;
      this.eventSource = new EventSourcePolyfill(url, {
        headers: {
          Token: this.config.token || '',
          'x-session-id': this.config.token || '',
          'x-app-code': this.config.appCode || '',
        },
        withCredentials: true,
        connectionTimeout: 60000,
        heartbeatTimeout: 60000,
      });
  
      this.eventSource.onopen = (event) => {
        // 移除这里的日志，只在外部处理器中打印
        if (this.handlers.onOpen) {
          this.handlers.onOpen(event);
        }
      };

      this.eventSource.addEventListener('message', (event) => {
        try {
          const data: SSEData = JSON.parse(event.data);
          
          // 只传递原始数据，模板处理在外部进行
          if (this.handlers.onMessage) {
            this.handlers.onMessage(data);
          }
        } catch (error) {
          console.error('处理SSE消息时出错:', error);
        }
      });

      // 监听keepalive事件
      this.eventSource.addEventListener('keepalive', (event) => {
        this.hasReceivedKeepalive = true; // 标记已收到keepalive
        console.log('💓 收到keepalive事件，连接正常');
      });

      this.eventSource.onerror = (error) => {
        console.error('SSE error:', error);
        
        // 检查是否为页面刷新导致的错误
        if (this.isPageRefreshing) {
          console.log('💡 页面刷新中，忽略SSE错误');
          return; // 页面刷新时忽略所有错误
        }
        
        // 检查是否为"No activity"错误 - 使用更宽松的检测条件
        const errorString = String(error).toLowerCase();
        
        // 检测各种可能的"No activity"错误变体
        const isNoActivityError = 
          errorString.includes('no activity') ||
          errorString.includes('no activity within') ||
          errorString.includes('milliseconds') && errorString.includes('reconnecting') ||
          errorString.includes('60000') && errorString.includes('reconnecting');
        
        // 如果是"No activity"错误且收到过keepalive，说明连接正常，忽略错误
        if (isNoActivityError && this.hasReceivedKeepalive) {
          this.hasReceivedKeepalive = false; // 重置状态，等待下一次keepalive
          return; // 不关闭连接，不触发重连，完全退出错误处理
        }
        
        // 如果只是"No activity"错误但没有收到keepalive，也尝试忽略（可能是误报）
        if (isNoActivityError) {
          return; // 不关闭连接，不触发重连
        }
        
        // 如果不是"No activity"错误，则执行正常的错误处理
        if (this.handlers.onError) {
          this.handlers.onError(error);
        }
        
        this.closeSSE();

        // 添加错误重连逻辑（仅在非页面刷新状态下）
        if (!this.isReconnecting.value && !this.isPageRefreshing) {
          setTimeout(() => {
            if (dialogSessionId) {
              this.reconnectSSE(dialogSessionId);
            }
          }, 3000);
        }
      };

    } catch (err) {
      console.error('初始化SSE连接失败:', err);
    }
  }

  // 重新连接SSE
  public reconnectSSE(newDialogSessionId: string): void {
    if (this.isReconnecting.value) {
      return;
    }
    this.isReconnecting.value = true;
    console.log('开始重连SSE，新的dialogSessionId:', newDialogSessionId);

    this.closeSSE();
    
    const reconnectTimeout = setTimeout(() => {
      this.initSSE(newDialogSessionId);
      setTimeout(() => {
        this.isReconnecting.value = false;
      }, 2000);
    }, 500);
    
    this.timeArr.push(reconnectTimeout);
    
    if (this.handlers.onReconnect) {
      this.handlers.onReconnect(newDialogSessionId);
    }
  }

  // 关闭SSE连接
  public closeSSE(): void {
    if (this.eventSource) {
      try {
        this.eventSource.close();
        this.eventSource = null;
      } catch (err) {
        console.warn('关闭SSE连接时出错:', err);
      }
    }

    // 清理所有定时器
    this.timeArr.forEach(timeout => {
      clearTimeout(timeout);
    });
    this.timeArr = [];
  }

  // 获取重连状态
  public getIsReconnecting(): boolean {
    return this.isReconnecting.value;
  }

  // 获取当前SSE连接状态
  public getConnectionState(): number {
    if (!this.eventSource) {
      return 0; // 未连接
    }
    return this.eventSource.readyState;
  }

  // 清理资源
  public destroy(): void {
    this.closeSSE();
    this.timeArr.forEach((item) => {
      clearTimeout(item);
    });
    this.timeArr = [];
    this.removePageRefreshListener();
  }

  // 设置页面刷新监听器
  private setupPageRefreshListener(): void {
    if (typeof window !== 'undefined') {
      // 监听页面卸载事件
      this.beforeUnloadHandler = () => {
        this.isPageRefreshing = true;
        console.log('💡 检测到页面刷新，主动关闭SSE连接');
        this.closeSSE();
      };
      // 添加多个事件监听器，确保能捕获各种页面关闭场景
      window.addEventListener('beforeunload', this.beforeUnloadHandler);
    }
  }

  // 移除页面刷新监听器
  private removePageRefreshListener(): void {
    if (typeof window !== 'undefined') {
      if (this.beforeUnloadHandler) {
        window.removeEventListener('beforeunload', this.beforeUnloadHandler);
        this.beforeUnloadHandler = null;
      }
    }
  }

  // 检查是否正在页面刷新
  public getIsPageRefreshing(): boolean {
    return this.isPageRefreshing;
  }
}

// 创建SSE服务实例的工厂函数
export function createSSEService(config: SSEServiceConfig, handlers: SSEHandlers = {}): SSEService {
  return new SSEService(config, handlers);
}

export default SSEService;