// 简化的微信静默登录工具类
class WeChatLogin {
  constructor() {
    this.appId = 'YOUR_WECHAT_APPID' // 替换为你的微信公众号AppID
    this.isWeChat = this.checkWeChatBrowser()
    this.isConfigured = this.appId !== 'YOUR_WECHAT_APPID'
  }

  // 检查是否为微信浏览器
  checkWeChatBrowser() {
    const ua = navigator.userAgent.toLowerCase()
    return ua.indexOf('micromessenger') !== -1
  }

  // 简化的静默登录
  async silentLogin() {
    // 如果未配置appId，使用模拟登录
    if (!this.isConfigured) {
      console.log('使用模拟静默登录')
      return this.mockLogin()
    }

    // 如果是微信浏览器，执行真实静默登录
    if (this.isWeChat) {
      return this.realWeChatLogin()
    }

    // 非微信浏览器，使用模拟登录
    console.log('非微信浏览器，使用模拟登录')
    return this.mockLogin()
  }

  // 真实微信静默登录
  realWeChatLogin() {
    return new Promise((resolve, reject) => {
      // 检查URL中是否已有授权code
      const urlParams = new URLSearchParams(window.location.search)
      const code = urlParams.get('code')

      if (code) {
        // 已有code，直接获取用户信息
        this.getUserInfo(code).then(resolve).catch(reject)
      } else {
        // 重定向到微信授权页面进行静默授权
        this.redirectToWeChat()
      }
    })
  }

  // 重定向到微信授权页面
  redirectToWeChat() {
    const redirectUri = encodeURIComponent(window.location.origin + window.location.pathname)
    const scope = 'snsapi_base' // 静默授权，不弹出授权页面
    const state = 'STATE_' + Date.now()
    
    const authUrl = `https://open.weixin.qq.com/connect/oauth2/authorize?appid=${this.appId}&redirect_uri=${redirectUri}&response_type=code&scope=${scope}&state=${state}#wechat_redirect`
    
    window.location.href = authUrl
  }

  // 获取用户信息（简化版）
  async getUserInfo(code) {
    try {
      // 这里需要调用后端接口来获取用户信息
      // 微信不允许前端直接调用获取用户信息的接口
      const response = await fetch('/api/wechat/userinfo', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ code })
      })

      if (!response.ok) {
        throw new Error('获取用户信息失败')
      }

      const userInfo = await response.json()
      
      // 保存用户信息到本地存储
      localStorage.setItem('wechat_user', JSON.stringify(userInfo))
      
      return {
        isLoggedIn: true,
        userInfo: userInfo,
        message: '微信静默登录成功'
      }
    } catch (error) {
      console.error('获取用户信息失败:', error)
      throw new Error('登录失败，请重试')
    }
  }

  // 模拟登录（用于开发和测试）
  mockLogin() {
    return new Promise((resolve) => {
      setTimeout(() => {
        const mockUser = {
          openid: 'mock_openid_' + Date.now(),
          nickname: '测试用户',
          headimgurl: '',
          isMock: true
        }
        
        localStorage.setItem('wechat_user', JSON.stringify(mockUser))
        
        resolve({
          isLoggedIn: true,
          userInfo: mockUser,
          message: '模拟登录成功'
        })
      }, 1000) // 模拟网络延迟
    })
  }

  // 检查登录状态
  checkLoginStatus() {
    const userData = localStorage.getItem('wechat_user')
    if (userData) {
      try {
        const userInfo = JSON.parse(userData)
        return {
          isLoggedIn: true,
          userInfo: userInfo
        }
      } catch (error) {
        console.error('解析用户数据失败:', error)
      }
    }
    
    return { isLoggedIn: false }
  }

  // 退出登录
  logout() {
    localStorage.removeItem('wechat_user')
  }

  // 获取当前用户信息
  getCurrentUser() {
    const status = this.checkLoginStatus()
    return status.isLoggedIn ? status.userInfo : null
  }
}

export default new WeChatLogin()