/**
 * Markdown模板工具类
 * 用于解析和渲染Markdown内容
 */

/**
 * 增强的Markdown解析器
 * @param text Markdown文本内容
 * @returns 解析后的HTML字符串
 */
export const parseMarkdown = (text: string): string => {
  // 确保text是字符串
  if (typeof text !== 'string') {
    text = String(text || '');
  }

  // 先进行HTML转义，防止XSS攻击
  text = text
    .replace(/&/g, '&amp;')
    .replace(/</g, '&lt;')
    .replace(/>/g, '&gt;')
    .replace(/"/g, '&quot;')
    .replace(/'/g, '&#x27;');

  // 然后处理表格（在HTML转义之后）
  text = text.replace(/(\|[^\n]+\|(?:\r?\n|\r))+/g, (match) => {
    const lines = match.trim().split(/\r?\n/).filter(line => line.trim() && line.includes('|'));
    if (lines.length < 1) return match;
    
    try {
      // 检查是否有分隔线
      let hasSeparator = false;
      let headerIndex = 0;
      let separatorIndex = -1;
      
      // 查找分隔线（第二行通常是分隔线）
      if (lines.length >= 2) {
        const secondLine = lines[1].trim();
        // 分隔线通常只包含 |、-、: 和空格
        if (/^[\s|:\-]+$/.test(secondLine.replace(/[^:\-|]/g, ''))) {
          hasSeparator = true;
          headerIndex = 0;
          separatorIndex = 1;
        }
      }
      
      // 解析表头
      const header = parseTableRow(lines[headerIndex]);
      
      // 确定列对齐方式
      const alignments = header.map(() => 'left'); // 默认左对齐
      if (hasSeparator && separatorIndex !== -1) {
        const separatorCells = parseTableRow(lines[separatorIndex]);
        separatorCells.forEach((cell, index) => {
          const content = cell.trim();
          if (content.startsWith(':') && content.endsWith(':')) {
            alignments[index] = 'center';
          } else if (content.startsWith(':')) {
            alignments[index] = 'left';
          } else if (content.endsWith(':')) {
            alignments[index] = 'right';
          }
        });
      }
      
      // 解析数据行
      const dataRows = [];
      const startIndex = hasSeparator ? 2 : 1;
      for (let i = startIndex; i < lines.length; i++) {
        dataRows.push(parseTableRow(lines[i]));
      }
      
      // 生成表格HTML
      let tableHtml = '<table class="markdown-table" style="border-collapse: collapse; width: 100%; border: 1px solid #ddd; margin: 10px 0;">\n';
      
      // 表头
      tableHtml += '  <thead>\n    <tr>\n';
      header.forEach((cell, index) => {
        const align = alignments[index] || 'left';
        tableHtml += `      <th style="text-align: ${align}; border: 1px solid #ddd; padding: 8px; background-color: #f5f5f5; font-weight: bold;">${cell}</th>\n`;
      });
      tableHtml += '    </tr>\n  </thead>\n';
      
      // 表体
      if (dataRows.length > 0) {
        tableHtml += '  <tbody>\n';
        dataRows.forEach(row => {
          tableHtml += '    <tr>\n';
          row.forEach((cell, index) => {
            const align = alignments[index] || 'left';
            tableHtml += `      <td style="text-align: ${align}; border: 1px solid #ddd; padding: 8px;">${cell}</td>\n`;
          });
          tableHtml += '    </tr>\n';
        });
        tableHtml += '  </tbody>\n';
      }
      
      tableHtml += '</table>';
      return tableHtml;
    } catch (error) {
      console.warn('表格解析失败:', error);
      return match;
    }
  });

  // 处理标题（支持1-6级）
  text = text.replace(/^######\s+(.*)$/gim, '<h6>$1</h6>');
  text = text.replace(/^#####\s+(.*)$/gim, '<h5>$1</h5>');
  text = text.replace(/^####\s+(.*)$/gim, '<h4>$1</h4>');
  text = text.replace(/^###\s+(.*)$/gim, '<h3>$1</h3>');
  text = text.replace(/^##\s+(.*)$/gim, '<h2>$1</h2>');
  text = text.replace(/^#\s+(.*)$/gim, '<h1>$1</h1>');

  // 处理粗体和斜体
  text = text.replace(/\*\*(.*?)\*\*/gim, '<strong>$1</strong>');
  text = text.replace(/\*(.*?)\*/gim, '<em>$1</em>');
  text = text.replace(/__(.*?)__/gim, '<strong>$1</strong>');
  text = text.replace(/_(.*?)_/gim, '<em>$1</em>');

  // 处理删除线
  text = text.replace(/~~(.*?)~~/gim, '<del>$1</del>');

  // 处理代码块（支持语言标识）
  text = text.replace(/```(\w+)?\n([\s\S]*?)```/gim, '<pre><code class="language-$1">$2</code></pre>');
  text = text.replace(/```([\s\S]*?)```/gim, '<pre><code>$1</code></pre>');
  text = text.replace(/`([^`]+)`/gim, '<code>$1</code>');

  // 处理链接（支持相对路径和绝对路径）
  text = text.replace(/\[([^\]]+)\]\(([^)]+)\)/gim, (match, text, url) => {
    // 验证URL格式
    const isValidUrl = /^(https?|ftp):\/\/[^\s/$.?#].[^\s]*$/i.test(url);
    const target = isValidUrl ? 'target="_blank" rel="noopener noreferrer"' : '';
    return `<a href="${url}" ${target}>${text}</a>`;
  });

  // 处理图片（添加加载失败处理）
  text = text.replace(/!\[([^\]]*)\]\(([^)]+)\)/gim, (match, alt, src) => {
    return `<img src="${src}" alt="${alt || '图片'}" style="max-width: 100%; height: auto;" onerror="this.style.display='none'" />`;
  });

  // 处理有序列表
  text = text.replace(/^\d+\.\s+(.*)$/gim, '<li>$1</li>');
  text = text.replace(/(<li>.*<\/li>)(?=\s*<li>)/gim, '$1');
  text = text.replace(/(<li>.*<\/li>)/gim, '<ol>$1</ol>');

  // 处理无序列表
  text = text.replace(/^[-*+]\s+(.*)$/gim, '<li>$1</li>');
  text = text.replace(/(<li>.*<\/li>)/gim, '<ul>$1</ul>');

  // 处理引用块
  text = text.replace(/^>\s+(.*)$/gim, '<blockquote>$1</blockquote>');

  // 处理水平分割线
  text = text.replace(/^\s*---\s*$/gim, '<hr />');
  text = text.replace(/^\s*\*\*\*\s*$/gim, '<hr />');

  // 完全删除段落处理逻辑，避免表格被p标签包裹
  // 只处理换行，不添加p标签
  text = text.replace(/\n{3,}/g, '\n\n'); // 多个换行合并为两个
  text = text.replace(/\n\n/g, '<br><br>');
  text = text.replace(/\n/g, '<br>');

  // 清理HTML结构（移除空行）
  text = text.replace(/(<br>\s*)+/g, '<br>');

  return text;
};

/**
 * 改进的表格行解析
 * @param line 表格行文本
 * @returns 单元格数组
 */
const parseTableRow = (line: string): string[] => {
  // 移除行首尾的 | 和空格
  const trimmedLine = line.trim().replace(/^\||\|$/g, '');
  // 按 | 分割，保留空单元格
  const cells = trimmedLine.split('|').map(cell => cell.trim());
  return cells;
};

/**
 * Markdown模板函数
 * @param content 要处理的Markdown内容
 * @returns 渲染后的HTML字符串
 */
export const markdownTemplate = (content: any): string => {
  // 类型检查和默认值处理
  if (typeof content !== 'string') {
    // 如果是对象，尝试转换为字符串
    if (content && typeof content === 'object') {
      content = JSON.stringify(content);
    } else {
      // 其他类型转换为字符串
      content = String(content || '');
    }
  }

  const htmlContent = parseMarkdown(content);

  // 清理HTML内容：移除br标签和空p段落
  const cleanHtml = htmlContent
    .trim()
    // 移除所有<br>标签
    .replace(/<br\s*\/?>/gi, '')
    // 移除空的<p>段落（只包含空格或换行符）
    .replace(/<p[^>]*>\s*<\/p>/gi, '')
    // 移除只包含空格的<p>段落
    .replace(/<p[^>]*>(\s|&nbsp;)*<\/p>/gi, '')
    // 移除连续的空白段落
    .replace(/(<\/p>\s*<p[^>]*>)+/gi, '')
    // 移除开头和结尾的空白段落
    .replace(/^\s*<p[^>]*>\s*<\/p>/gi, '')
    .replace(/<p[^>]*>\s*<\/p>\s*$/gi, '')
    .trim();

  // 检查内容是否为空或只有空白
  if (!cleanHtml || cleanHtml === '<p></p>' || cleanHtml === '<p>&nbsp;</p>') {
    return ''; // 如果内容为空，返回空字符串不展示
  }

  return `<div class="message-markdown">
    <div class="markdown-content">${cleanHtml}</div>
  </div>`;
};

/**
 * 简化的Markdown生成函数（兼容旧版本）
 * @param content Markdown内容
 * @returns 渲染后的HTML字符串
 */
export const markdown = (content: any): string => {
  return markdownTemplate(content);
};

/**
 * 检查内容块是否是markdown块
 * @param contentBlock 内容块对象
 * @returns 是否是markdown块
 */
export const isMarkdownBlock = (contentBlock: any): boolean => {
  return contentBlock && contentBlock.content && contentBlock.content.includes('message-markdown');
};

/**
 * 检查最后一个contentBlock是否是markdown块
 * @param contentBlocks 内容块数组
 * @returns 最后一个块是否是markdown块
 */
export const isLastBlockMarkdown = (contentBlocks: any[]): boolean => {
  if (!contentBlocks || contentBlocks.length === 0) {
    return false;
  }
  const lastBlock = contentBlocks[contentBlocks.length - 1];
  return isMarkdownBlock(lastBlock);
};

/**
 * 获取最后一个markdown块的索引
 * @param contentBlocks 内容块数组
 * @returns 最后一个markdown块的索引，如果没有找到返回-1
 */
export const getLastMarkdownBlockIndex = (contentBlocks: any[]): number => {
  if (!contentBlocks || contentBlocks.length === 0) {
    return -1;
  }
  for (let i = contentBlocks.length - 1; i >= 0; i--) {
    if (isMarkdownBlock(contentBlocks[i])) {
      return i;
    }
  }
  return -1;
};

/**
 * 合并markdown内容
 * @param existingContent 现有的markdown内容
 * @param newContent 新的markdown内容
 * @returns 合并后的markdown内容
 */
export const mergeMarkdownContent = (existingContent: string, newContent: string): string => {
  // 从现有的markdown内容中提取内部内容
  const existingInnerContent = existingContent.replace(/<div class="message-markdown">\s*<div class="markdown-content">([\s\S]*?)<\/div>\s*<\/div>/, '$1');

  // 从新的markdown内容中提取内部内容
  const newInnerContent = newContent.replace(/<div class="message-markdown">\s*<div class="markdown-content">([\s\S]*?)<\/div>\s*<\/div>/, '$1');

  // 合并内容并重新包裹
  const mergedInnerContent = existingInnerContent + newInnerContent;

  return `<div class="message-markdown">
    <div class="markdown-content">${mergedInnerContent}</div>
  </div>`;
};

export default {
  parseMarkdown,
  markdownTemplate,
  markdown,
  isMarkdownBlock,
  isLastBlockMarkdown,
  getLastMarkdownBlockIndex,
  mergeMarkdownContent
};