/**
 * 音频模板工具类
 * 用于生成音频消息的HTML模板和音频播放器管理
 */

/**
 * 音频消息模板 - 简化版本，移除audio-icon
 * @param audioData 音频数据对象
 * @returns 音频消息的HTML字符串
 */
export const audioTemplate = (audioData: any): string => {
  const { audioUrl, audioBlob, durationTime } = audioData;
  let src = audioUrl;

  // 如果提供了Blob对象，创建对象URL
  if (audioBlob && !audioUrl) {
    src = URL.createObjectURL(audioBlob);
  }

  // 生成唯一ID用于音频播放器
  const audioId = `audio_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

  return `<div class="audio-message" data-audio-id="${audioId}">
    <div class="audio-player" data-audio-src="${src}">
      <div class="audio-wave">
        <div class="wave-bar"></div>
        <div class="wave-bar"></div>
        <div class="wave-bar"></div>
        <div class="wave-bar"></div>
        <div class="wave-bar"></div>
      </div>
      <div class="audio-duration">${durationTime+'"' || '0"'}</div>
    </div>
    <audio id="${audioId}" src="${src}" preload="metadata" style="display: none;"></audio>
  </div>`;
};

/**
 * 初始化音频播放器
 * 设置音频播放器的事件监听
 */
export const initAudioPlayers = (): void => {
  const audioPlayers = document.querySelectorAll('.audio-player');

  audioPlayers.forEach((player) => {
    // 检查是否已经绑定过事件监听器
    if (player.hasAttribute('data-audio-bound')) {
      return; // 如果已经绑定过，跳过
    }

    // 标记为已绑定
    player.setAttribute('data-audio-bound', 'true');

    const audioMessage = player.closest('.audio-message');
    const audioId = audioMessage?.getAttribute('data-audio-id');
    const audioElement = audioId ? document.getElementById(audioId) : null;

    if (!audioElement) {
      console.warn('未找到音频元素，audioId:', audioId);
      return;
    }

    // 音频播放结束，重置状态
    audioElement.addEventListener('ended', () => {
      player.classList.remove('playing');
    });

    // 设置播放/暂停事件
    player.addEventListener('click', (e) => {
      e.stopPropagation();

      if (audioElement.paused) {
        // 暂停其他正在播放的音频
        pauseAllOtherAudios(audioElement);
        audioElement.play().catch(error => {
          console.error('播放音频失败:', error);
        });
        player.classList.add('playing');
      } else {
        audioElement.pause();
        player.classList.remove('playing');
      }
    });

    // 音频播放事件
    audioElement.addEventListener('play', () => {
      player.classList.add('playing');
    });

    // 音频暂停事件
    audioElement.addEventListener('pause', () => {
      player.classList.remove('playing');
    });
  });
};

/**
 * 暂停所有其他正在播放的音频
 * @param currentAudio 当前正在播放的音频元素
 */
export const pauseAllOtherAudios = (currentAudio: HTMLAudioElement): void => {
  const allAudios = document.querySelectorAll('audio');
  allAudios.forEach((audio) => {
    if (audio !== currentAudio && !audio.paused) {
      audio.pause();
      const player = audio.closest('.audio-message')?.querySelector('.audio-player');
      if (player) {
        player.classList.remove('playing');
      }
    }
  });
};

/**
 * 简化的音频模板函数（兼容旧版本）
 * @param audioData 音频数据对象
 * @returns 音频消息的HTML字符串
 */
export const audio = (audioData: any): string => {
  return audioTemplate(audioData);
};

// 默认导出对象
export default {
  audioTemplate,
  audio,
  initAudioPlayers,
  pauseAllOtherAudios
};