/**
 * 表格模板工具类
 * 用于生成和渲染数据表格
 */

// 表格数据接口
export interface TableData {
  [key: string]: any;
}

// 表格配置接口
export interface TableConfig {
  showFooter?: boolean;
}

/**
 * 生成表格HTML
 * @param tableData 表格数据数组
 * @param config 表格配置
 * @returns 表格HTML字符串
 */
export const generateTableHTML = (tableData: TableData[], config: TableConfig = {}): string => {
  const { showFooter = true } = config;

  // 处理空数据
  if (!Array.isArray(tableData) || tableData.length === 0) {
    return `
      <div class="message-table">
        <div class="table-title">数据表格</div>
        <div class="table-empty">暂无数据</div>
      </div>`;
  }

  // 动态生成表头 - 使用第一条数据的键名
  const headers = Object.keys(tableData[0]);

  // 判断列是否为数字列
  const isNumericColumn = (header: string) => {
    return tableData.some(row => {
      const value = row[header];
      return typeof value === 'number' || (!isNaN(parseFloat(value)) && isFinite(value));
    });
  };

  // 数字格式化函数 - 万/亿格式化，保留两位小数，向上取整
  const formatNumber = (value: any) => {
    if (value === null || value === undefined || value === '') return '';

    const num = parseFloat(value);
    if (isNaN(num)) return value;

    if (num >= 100000000) {
      // 亿级别
      const result = Math.ceil((num / 100000000) * 100) / 100;
      return result.toFixed(2) + '亿';
    } else if (num >= 10000) {
      // 万级别
      const result = Math.ceil((num / 10000) * 100) / 100;
      return result.toFixed(2) + '万';
    } else {
      // 小于万
      return Math.ceil(num).toString();
    }
  };

  // 处理单元格内容，特别处理趋势列和数字列
  const renderCellContent = (header: string, value: any) => {
    // 如果是趋势列，根据up/down值显示箭头图标
    if (header.toLowerCase().includes('趋势') || header.toLowerCase().includes('trend')) {
      const trendValue = String(value).toLowerCase().trim();
      if (trendValue === 'up' || trendValue === '上升' || trendValue === '上涨') {
        return `<span class="trend-up">↑</span>`;
      } else if (trendValue === 'down' || trendValue === '下降' || trendValue === '下跌') {
        return `<span class="trend-down">↓</span>`;
      }
    }

    // 如果是数字列，进行格式化
    if (isNumericColumn(header)) {
      return formatNumber(value);
    }

    // 其他列保持原样
    return value;
  };

  // 判断列是否为趋势列
  const isTrendColumn = (header: string) => {
    return header.toLowerCase().includes('趋势') || header.toLowerCase().includes('trend');
  };

  // 为不同列类型添加对应的样式类
  const getCellClass = (header: string) => {
    if (isTrendColumn(header)) {
      return 'trend-cell'; // 趋势列居中
    } else if (isNumericColumn(header)) {
      return 'numeric-cell'; // 数字列右对齐
    } else {
      return 'text-cell'; // 文字列左对齐
    }
  };

  return `
    <div class="message-table">
      <div class="table-container">
        <table class="data-table">
          <thead>
            <tr>
              ${headers.map(header => `<th class="${getCellClass(header)}">${header}</th>`).join('')}
            </tr>
          </thead>
          <tbody>
            ${tableData.map(row => `
              <tr>
                ${headers.map(header => `
                  <td class="${getCellClass(header)}">
                    ${renderCellContent(header, row[header])}
                  </td>
                `).join('')}
              </tr>
            `).join('')}
          </tbody>
        </table>
      </div>
      ${showFooter ? `<div class="table-footer">共 <span>${tableData.length}</span> 条记录</div>` : ''}
    </div>
  `;
};

/**
 * 简化的表格生成函数（兼容旧版本）
 * @param tableData 表格数据
 * @returns 表格HTML字符串
 */
export const tableTemplate = (tableData: any): string => {
  return generateTableHTML(tableData);
};

export default {
  generateTableHTML,
  tableTemplate
};